Rem
Rem $Header: plscopedemo.sql 04-jan-2007.13:15:40 rdecker Exp $
Rem
Rem plscopedemo.sql
Rem
Rem Copyright (c) 2006, 2007, Oracle. All rights reserved.  
Rem
Rem    NAME
Rem      plscopedemo.sql - Demo program for PL/Scope
Rem
Rem    DESCRIPTION
Rem      This is a simple web browser based sample program to demonstrate 
Rem      the interfaces of PL/Scope, the PL/SQL identifier collection and 
Rem      cross-reference mechanism.
Rem
Rem    NOTES
Rem      This demo requires either mod_plsql or the embedded PL/SQL 
Rem      gateway (including Oracle XML DB) be installed.  In addition, the 
Rem      SCOTT demo account must exist along with a database access 
Rem      descriptor (DAD) for this account.  Please refer to the mod_plsql
Rem      or embedded PL/SQL gateway documentation for instructions on how
Rem      to create a DAD.
Rem      If you are using the embedded PL/SQL gateway, you may also refer
Rem      to the embedded PL/SQL gateway demo (epgdemo.sql) for an
Rem      embedded gateway demonstration.
Rem
Rem      To execute this demo, after the DAD is created, load the plscopedemo 
Rem      script:
Rem
Rem      % sqlplus /nolog @plscopedemo
Rem  
Rem      Then, open http://<your-database-host>:<http-port>/<dad>/plscope.main
Rem
Rem    MODIFIED   (MM/DD/YY)
Rem    rdecker     11/02/06 - Created
Rem

CONNECT scott/tiger

Rem The PLSCOPE_SETTINGS parameter must be set to 'identifiers:all' to collect
Rem PL/SQL identifiers.
alter session set PLSCOPE_SETTINGS='identifiers:all';

CREATE OR replace PACKAGE PLScope IS
   -- Setup the Main screen, which builds the frames used in the demo
   PROCEDURE Main; 

   -- Display the Banner/welcome screen
   PROCEDURE Banner;

   -- Initialize a frame display name
   PROCEDURE FrameTitle(p_name varchar2);

   -- Find an identifier with the given signature
   PROCEDURE DataHit(p_sig varchar2);

   -- Find all identifiers that match the given identifier name
   PROCEDURE SrcHit(p_sig varchar2);

   -- Display the source code of object name, type and owner, and highlight
   -- identifiers with the matching signature.
   PROCEDURE DisplaySource(p_objname varchar2, p_objtype varchar2, 
                           p_owner varchar2, p_sig varchar2);

   -- Create a form for looking up identifiers by name
   PROCEDURE HitBox;

   -- Display the HitBox results
   PROCEDURE HitList(p_name VARCHAR2, p_type varchar2, p_usage varchar2, 
                     p_submit varchar2 default Null);
END PLScope;
/

CREATE OR replace PACKAGE BODY PLScope IS

   -- Create a table of all PL/SQL types known to PL/Scope
   type IdentifierType is table of varchar2(35);

   AllTypes IdentifierType := 
     IdentifierType('ASSOCIATIVE ARRAY','BFILE DATATYPE','BLOB DATATYPE','BOOLEAN DATATYPE',
                    'CHARACTER DATATYPE','CLOB DATATYPE','CONSTANT','CURSOR','DATE DATATYPE',
                    'EXCEPTION','FORMAL IN','FORMAL IN OUT','FORMAL OUT','FUNCTION',
                    'INDEX TABLE','INTERVAL DATATYPE','ITERATOR','LABEL','LIBRARY','NESTED TABLE',
                    'NUMBER DATATYPE','OBJECT','OPAQUE','PACKAGE','PROCEDURE','RECORD','REFCURSOR',
                    'SUBTYPE','SYNONYM','TIME DATATYPE','TIMESTAMP DATATYPE','TRIGGER','UROWID',
                    'VARIABLE','VARRAY');

   -- Get an identifier declaration.
   -- If passed a signature, this cursor will return one row containing the
   -- declaration of the identifier with the given signature.
   -- If passed a name, this cursor will return rows containing declarations for
   -- all identifiers whose names match p_name.
   cursor GetIdentDeclCursor (p_name VARCHAR2, p_sig varchar2) IS
     SELECT *
       FROM all_identifiers
       WHERE (name like p_name or name like upper(p_name) or signature=p_sig) and
             usage = 'DECLARATION'
       ORDER by name, type;
     
   -- Display the Banner/welcome screen
   PROCEDURE Banner IS
   BEGIN
      htp.p('<h3><center>Welcome to the PL/Scope Prototype Demo</center></h3>');
      htp.br;
      htp.p('What is PL/Scope?  PL/Scope is an source code identifier scoping tool for PL/SQL, much like cscope.  However, unlike cscope, PL/Scope symbols are generated from within the PL/SQL compiler, allowing all identifiers (even local variables!) and overloaded functions/procedures to be scoped with accuracy - something that cannot be achieved in a post-compilation tool such as cscope.  The PL/Scope identifier data can be accessed by all applications using dba/user/all views and should prove to be extremely useful in IDE environments.');
      htp.br;
      htp.p('The following demo is meant to give users a taste of what PL/Scope has to offer, and to demonstrate that even a lightweight, web-based code browser can provide great insight into PL/SQL code design.');
      htp.br;
      htp.p('PL/Scope does not collect identifiers from PL/SQL code blocks by default.  Instead, the PLSCOPE_SETTINGS parameter must be set to ''identifiers:all'' before a PL/SQL code block is compiled or recompiled. This parameter is set when this demo is compiled.');
      htp.br;
      htp.p('<b>To begin the demo</b>, enter PLSCOPE into the "Search Lookup" text box and click on PACKAGE when the search results appear. Then click on ALL_USAGES in the "Description" window and finally click on DECLARATION  or DEFINITION in the "Usages" window to begin browsing the PLScope package.  Clicking on symbols in the Source code will update only the middle "Description" window, and you must click on ALL_USAGES to show all uses of that identifier followed by clicking on a usage to display locations in the code where those usages appear.');
      htp.br;
   END;
    
   -- Build and display the Main screen, complete with frames 
   PROCEDURE Main IS
   BEGIN
      htp.title('PL/Scope Demo');
      htp.framesetopen('65%, 35%');
      htp.frame('PLScope.Banner','srcwindow');
      htp.framesetopen(NULL,'25%,35%,40%');
      htp.frame('PLScope.HitBox','hitwindow');
      htp.frame('PLScope.FrameTitle?p_name=Description Window','datawindow');
      htp.frame('PLScope.FrameTitle?p_name=Usages Window','usagewindow');
      htp.framesetclose;
      htp.framesetclose;
   END;

   -- Print title of frame
   PROCEDURE FrameTitle(p_name varchar2) is
   begin
     htp.p('<b><center>'||p_name||'</center></b>');
   end;
   
   -- Display the description of an identifier
   PROCEDURE DisplayDescription(Identifier GetIdentDeclCursor%ROWTYPE) 
   is
      datatype_name       VARCHAR2(100);
      datatype_signature  varchar2(32);
      datatype_usage      varchar2(30);
      datatype_usage_id   number;
      Context             all_identifiers%rowtype;
   BEGIN
      -- If this identifier declaration has a usage_context_id, get it.
      -- This will tell us in what context the identifer declaration 
      -- was done.  For instance, package/procedure/fuction/etc.
      -- If a declaration was done at the top-level (for instance, this
      -- is a package declaration) then the context number will be 0.
      if (Identifier.usage_context_id > 0) then
        begin
          SELECT *
            INTO Context
            FROM all_identifiers 
            WHERE usage_id    = Identifier.usage_context_id and 
                  object_name = Identifier.object_name      and 
                  object_type = Identifier.object_type      and
                  owner       = Identifier.owner;
        exception
        when no_data_found then
          null;
        end;
      end if;
      
      htp.p('<b><center>Description of "'||replace(Identifier.name, '<', '&#60;')||'"</center></b>');

      -- If the identifier is a synonym, then we need to display the
      -- information differently.  The reason is that a synonym declaration has
      -- no source code, so we have to "dereference" it to find
      -- displayable source code.
      if (Identifier.type = 'SYNONYM') then
        declare
          syn_source   all_identifiers%rowtype;
        begin
 
          -- Get the signature of the synonym "source" using all_synonyms
          -- This is essentially the signature of the identifier in the FOR clause
          -- of the create synonym statement.
          select idents.*
            INTO syn_source
            FROM all_identifiers idents, 
                 (select table_name name, table_owner owner 
                    FROM all_synonyms 
                    WHERE synonym_name = Identifier.name and 
                          owner        = Identifier.owner) syns 
            WHERE idents.name  = syns.name  and 
                  idents.owner = syns.owner and 
                  usage        = 'DECLARATION';

          -- Display a link to the source of the synonym
          htp.anchor2('PLScope.DisplaySource?p_objname='||syn_source.object_name||
                      chr(38)||'p_objtype='||syn_source.object_type||chr(38)||
                      'p_owner='||syn_source.owner||chr(38)||'p_sig='||
                      syn_source.signature||
                      '#'||syn_source.line,'<b>SOURCE DECLARATION</b>', 'actHitList',
                      'srcwindow');
          htp.p('     ');
        exception
          -- This can happen if there is no source for the identifier - for example, 
          -- if the synonym if for a view.  There are cases of this in this file:
          -- i.e.   SType all_identifiers%rowtype;
          -- all_identifiers is a synonym for a view, therefore, we can't display the source.
          when others then
            null;
        end;

      else
        -- Display a quick link to the declaration in the source
             htp.anchor2('PLScope.DisplaySource?p_objname='||Identifier.object_name||
                    chr(38)||'p_objtype='||Identifier.object_type||chr(38)||
                    'p_owner='||Identifier.owner||chr(38)||'p_sig='||
                    Identifier.signature||
                    '#'||Identifier.line,'<b>DECLARATION</b>', 'actHitList','srcwindow');
        htp.p('     ');
      end if;

      -- Display a quick link to the definition if there is one
      if (Identifier.type in ('FUNCTION', 'PROCEDURE', 'PACKAGE', 'OBJECT'))
      then
         declare
           object_type varchar2(30);
           line        number;
         begin
           SELECT object_type, line
             INTO object_type, line
             FROM all_identifiers
             WHERE signature=Identifier.signature and
                   usage = 'DEFINITION';

             htp.anchor2('PLScope.DisplaySource?p_objname='||Identifier.object_name||
                  chr(38)||'p_objtype='||object_type||chr(38)||
                  'p_owner='||Identifier.owner||chr(38)||'p_sig='||
                  Identifier.signature||
                  '#'||line,'<b>DEFINITION</b>', 'actHitList','srcwindow');
             htp.p('     ');

          -- it is possible that this data was not collected
          exception
          when no_data_found then
            null;
          end;
        end if;

      -- Display a link to show all usages of this identifier
      htp.anchor2('PLScope.SrcHit?p_sig='|| 
        Identifier.signature,'<b>ALL USAGES</b>','declHitList', 'usagewindow');
      htp.br;

      -- Now display informational fields
      htp.p('<b>OWNER: </b>'|| Identifier.owner);
      htp.br;

      -- In what context is the identifier declared?
      if (Identifier.usage_context_id > 0) then
         htp.p('<b>'||Identifier.type||'</b> IN ');

         if (Context.type in ('PACKAGE', 'OBJECT')) then
           htp.p('<b>'||Context.object_type||'</b> ');
         else
           htp.p('<b>'||Context.type||'</b> ');
         end if;
         htp.anchor2('PLScope.DataHit?p_sig='||Context.signature,
                     Context.name, 'declHitList', 'datawindow');
         htp.br;
      else
        htp.p('<b>TOP-LEVEL '||Identifier.type||'</b>');
        htp.br;
      end if;

      -- Print out any associated datatypes for variable or formal types
      if (Identifier.type in ('VARIABLE','FORMAL IN','FORMAL IN OUT',
                              'FORMAL OUT', 'CONSTANT')) 
      then
        begin
          -- Find the datatype of the variable (including formals)
          select signature, name
            into datatype_signature, datatype_name
            from all_identifiers
            where usage_context_id = Identifier.usage_id    and 
                  object_name      = Identifier.object_name and 
                  object_type      = Identifier.object_type and 
                  owner            = Identifier.owner       and
                  usage            = 'REFERENCE';

          htp.p('<b>DATATYPE: </b>');
          htp.anchor2('PLScope.SrcHit?p_sig='|| 
            datatype_signature,datatype_name,'declHitList', 'usagewindow');
          htp.br;

        -- It is possible that this data was not collected
        exception
        when no_data_found then
          null;
        end;
      end if;

      -- Print out the return type if this is a function
      if (Identifier.type = 'FUNCTION') then
        declare
          -- The first row of this cursor will either be
          -- A. The definition of a function or
          -- B. The return type of a function
          -- Ultimately, what we want is the first reference (smallest usage_id)
          -- in the context of a declaration or definition.  We do this by 
          -- fetching only the first row retrieved by this cursor.
          cursor GetReturnType(p_usage_id number, p_name varchar2, p_type varchar2, 
                               p_owner varchar2) is
            select signature, name, usage, usage_id 
              from all_identifiers
              where usage_context_id = p_usage_id             and 
                    (usage='REFERENCE' or usage='DEFINITION') and
                    object_name      = p_name                 and 
                    object_type      = p_type                 and 
                    owner            = p_owner
              order by usage_id;
        begin
          open GetReturnType(Identifier.usage_id, Identifier.object_name,
                             Identifier.object_type, Identifier.owner);
          fetch GetReturnType 
            into datatype_signature, datatype_name, datatype_usage, datatype_usage_id;
          close GetReturnType;

          -- Functions have both a declaration and definition.  These two can occur
          -- such that the declaration is the usage_context_id of the definition, and if this
          -- is the case, then we need to repeat the query one more time to find the
          -- datatype reference which will be to the definition.
          if (datatype_usage = 'DEFINITION') then
            open GetReturnType(datatype_usage_id, Identifier.object_name, 
                               Identifier.object_type, Identifier.owner);
            fetch GetReturnType
              into datatype_signature, datatype_name, datatype_usage, datatype_usage_id;
            close GetReturnType;
          end if;

          htp.p('<b>RETURNS: </b>');
          htp.anchor2('PLScope.SrcHit?p_sig='|| 
            datatype_signature,datatype_name,'declHitList', 'usagewindow');
          htp.br;

        -- It is possible that this data was not collected
        exception
        when no_data_found then
          null;
        end;
      end if; 

      -- List out the arguments for functions, procedures and cursors
      if (Identifier.type in ('FUNCTION', 'PROCEDURE', 'CURSOR')) then
        htp.p('<b>ARGUMENTS: </b>');

        declare
          cursor GetArgumentsCursor(p_usage_id number) is
            select name, signature
              from all_identifiers
              where type like 'FORMAL %' and 
                    usage_context_id = p_usage_id and
                    object_name      = Identifier.object_name and 
                    object_type      = Identifier.object_type and 
                    owner            = Identifier.owner and 
                    usage            = 'DECLARATION'
              order by usage_id;

          Argument GetArgumentsCursor%ROWTYPE;
          FoundArg boolean := FALSE;
          UsageID  number;
        begin
          -- First let's see if this is a declaration which has a definition at 
          -- the same location.  If so, then we need to use the definition because
          -- the definition will be in the usage_context_id of the declaration and the
          -- arguments will be in the usage_context_id of the definition.
          begin
            select usage_id 
              into UsageID
              from all_identifiers
              where usage_context_id = Identifier.usage_id and 
                    object_name      = Identifier.object_name and 
                    object_type      = Identifier.object_type and 
                    owner            = Identifier.owner and 
                    usage            = 'DEFINITION';
          exception
          when no_data_found then
            -- No definition found, so use the declaration usage_id as the usage_context_id
            UsageID := Identifier.usage_id;
          end;

          for Argument in GetArgumentsCursor(UsageID) loop
            htp.br;
            htp.p('..');
            htp.anchor2('PLScope.DataHit?p_sig='|| Argument.signature,
              Argument.name, 'declHitList', 'datawindow');
            FoundArg := TRUE;
          end loop;
      
          if (FoundArg = FALSE) then
            htp.p('NONE');
          end if;
          htp.br;
        end;
      end if;

      -- Get and display the context of a subtype
      if (Identifier.type in ('SUBTYPE')) then
        declare
          SType all_identifiers%rowtype;
        begin
          select *
            into SType
            from all_identifiers
            where usage_context_id = Identifier.usage_id and
                  object_name      = Identifier.object_name and 
                  object_type      = Identifier.object_type and 
                  owner            = Identifier.owner and 
                  usage            = 'REFERENCE';

          htp.p('<b>SUBTYPE </b> OF ');
          htp.anchor2('PLScope.DataHit?p_sig='|| SType.signature,
                SType.name, 'declHitList', 'datawindow');

          -- There is a reference to Package/Object.type.  Repeat the query to get 
          -- the type.
          if (SType.type in ('PACKAGE', 'OBJECT')) then
            select *
              into SType
              from all_identifiers
              where usage_context_id = SType.usage_id and
                    object_name      = SType.object_name and 
                    object_type      = SType.object_type and 
                    owner            = SType.owner and 
                    usage            = 'REFERENCE';

            htp.p('.');
            htp.anchor2('PLScope.DataHit?p_sig='|| SType.signature,
                  SType.name, 'declHitList', 'datawindow');
          end if;

        -- It is possible that this data was not collected
        exception
        when no_data_found then
          null;
        end;
 
      end if;
        
      -- List out Record Field variables
      if (Identifier.type in ('RECORD')) then
        declare
          cursor GetFieldsCursor(p_usage_id number) is
            select name, signature
              from all_identifiers
              where usage_context_id = p_usage_id and
                    object_name      = Identifier.object_name and 
                    object_type      = Identifier.object_type and 
                    owner            = Identifier.owner
              order by usage_id;
        begin
          htp.p('FIELDS:');
          htp.br;

          for Field in GetFieldsCursor(Identifier.usage_id) loop
            htp.p('..');
            htp.anchor2('PLScope.DataHit?p_sig='|| Field.signature,
                        Field.name, 'declHitList', 'datawindow');
            htp.br;
          end loop;
        end;
      end if;

      -- Describe collection types
      if (Identifier.type in ('NESTED TABLE', 'VARRAY', 'ASSOC ARRAY', 'INDEX TABLE'))
      then
        declare
          Collection all_identifiers%rowtype;
          Collection2 all_identifiers%rowtype;
        begin
          select *
            into Collection
            from all_identifiers
            where usage_context_id = Identifier.usage_id and
                  object_name      = Identifier.object_name and 
                  object_type      = Identifier.object_type and 
                  owner            = Identifier.owner;

          -- Nested tables and varrays have only a collection type
          if (Identifier.type in ('NESTED TABLE', 'VARRAY')) then
            htp.p('<b>COLLECTION </b>OF ');

            htp.anchor2('PLScope.DataHit?p_sig='|| Collection.signature, 
                        Collection.name, 'declHitList', 'datawindow');
            htp.br;
          else
            -- Index tables also have an index type
            begin
              select *
                into Collection2
                from all_identifiers
                where usage_context_id = Collection.usage_id and
                      object_name      = Collection.object_name and 
                      object_type      = Collection.object_type and 
                      owner            = Collection.owner;

                htp.p('<b>COLLECTION </b>OF ');
                htp.anchor2('PLScope.DataHit?p_sig='|| Collection.signature, 
                            Collection.name, 'declHitList', 'datawindow');
                htp.br;
                htp.p('<b>INDEX BY</b> ');
                htp.anchor2('PLScope.DataHit?p_sig='|| Collection2.signature,
                            Collection2.name, 'declHitList', 'datawindow');
                htp.br;
            exception 
            when no_data_found then
              if (Collection.type in ('VARCHAR2', 'BINARY_INTEGER')) then
                htp.p('<b>INDEX BY</b> ');
                htp.anchor2('PLScope.DataHit?p_sig='|| Collection.signature,
                            Collection.name, 'declHitList', 'datawindow');
                htp.br;
              else
                htp.p('<b>COLLECTION </b>OF ');
                htp.anchor2('PLScope.DataHit?p_sig='|| Collection.signature,
                            Collection.name, 'declHitList', 'datawindow');
                htp.br;
              end if;
            end;
          end if;   
        exception 
        when no_data_found then
          null;
        end;
      end if;

   exception
   when others then
      htp.p('Unhandled Exception: '||sqlerrm);

   END;
      
   -- Display an identifier in the search window
   PROCEDURE DisplayIdentifierList(Identifier GetIdentDeclCursor%ROWTYPE)
   IS
   BEGIN
      htp.p('..');
      htp.anchor2('PLScope.DataHit?p_sig='|| Identifier.signature,
        Identifier.type,'declHitList', 'datawindow');
      htp.br;
   END;
   

   -- Display All Usages in the Usage window   
   PROCEDURE DisplayAllList(IdentifierDecl GetIdentDeclCursor%ROWTYPE)
   IS
      cursor actcur(p_sig varchar2) IS 
        SELECT usage, line, col, object_name, object_type, owner, type
          FROM all_identifiers
          WHERE signature = p_sig
          ORDER by usage;

        acrow actcur%ROWTYPE;        
        objname VARCHAR2(30);
        objown  VARCHAR2(30);
        objtype varchar2(30);
        found   boolean := FALSE;
   BEGIN
      htp.p('<b><center>Usages of ' || IdentifierDecl.type || ' "' ||
            replace(IdentifierDecl.name, '<', '&#60;') || 
            '"</center></b>');

      -- Loop through and display all usages of the identifier with the
      -- given signature.
      FOR acrow IN actcur(IdentifierDecl.signature) LOOP
         -- Do not display synonym declarations because they don't have
         -- any associated source code.
         if (not (acrow.type = 'SYNONYM' and acrow.usage = 'DECLARATION')) then
           htp.anchor2('PLScope.DisplaySource?p_objname='||acrow.object_name||
                        chr(38)||'p_objtype='||acrow.object_type||chr(38)||
                        'p_owner='||acrow.owner||chr(38)||'p_sig='||
                         IdentifierDecl.signature||
                        '#'||acrow.line,acrow.usage,'actHitList','srcwindow');


           htp.p(' at '|| acrow.owner || '.' || acrow.object_name || ' line ' || 
             acrow.line ||', col '|| acrow.col);
           htp.br;
           found := TRUE;
         end if;
      END LOOP;
      
      -- Didn't find a usage.  This could happen in certain situations, such as 
      -- when looking for usages of synonyms.  Synonym declarations do not contain
      -- source (they are skipped above), so beyond that, if the synonym is not
      -- used we could get this condition.
      if (found = FALSE) then
        htp.br;
        htp.p('<b><center STYLE="color:red>No usages found</center></b>');
      end if;
      htp.br;

   END;
      
   -- Display an identifier description using the identifier signature
   PROCEDURE DataHit(p_sig varchar2) IS
      Identifier GetIdentDeclCursor%ROWTYPE;
   BEGIN
      open GetIdentDeclCursor(NULL, p_sig);
      fetch GetIdentDeclCursor INTO Identifier;
      DisplayDescription(Identifier);
      close GetIdentDeclCursor;
   END;
   
   -- Display all usages of an identifier using the identifier signature
   PROCEDURE SrcHit(p_sig varchar2) IS
      Identifier GetIdentDeclCursor%ROWTYPE;
   BEGIN
      open GetIdentDeclCursor(NULL, p_sig);
      fetch GetIdentDeclCursor INTO Identifier;
      DisplayAllList(Identifier);
      close GetIdentDeclCursor;
   END;

   -- Determine the color of an identifier based on its type
   Function GetTextColor(p_type varchar2) return varchar2 is
   begin
     if (p_type in ('VARIABLE', 'ITERATOR') or p_type like 'FORMAL %') then
       return 'red';
     elsif (p_type in ('PROCEDURE', 'FUNCTION', 'CURSOR')) then
       return 'green';
     elsif (p_type in ('PACKAGE', 'PACKAGE BODY', 'OBJECT', 'OBJECT BODY')) then
       return 'blue';
     elsif (p_type like '% DATATYPE') then
       return 'orange';
     elsif (p_type in ('ASSOC ARRAY', 'INDEX TABLE', 'NESTED TABLE', 'VARRAY')) then
       return 'purple';
     elsif (p_type in ('EXCEPTION', 'CONSTANT')) then
       return 'brown';
     else
       return 'gray';
     end if;
   end;
      
   -- Display the source code
   PROCEDURE DisplaySource(p_objname varchar2, p_objtype varchar2, 
                           p_owner varchar2, p_sig varchar2)
   is
      SrcCol     pls_integer;
      PrintChars pls_integer;
      ident_end  pls_integer;
      
      -- Cursor to find all identifiers
      cursor GetIdentifiersCursor IS 
        SELECT distinct name,signature,line,col,type
         FROM all_identifiers
         WHERE object_name=p_objname and object_type=p_objtype
               and owner=p_owner;

      -- Get Text from all_source
      cursor GetTextLineCursor is
        SELECT text,line
         FROM all_Source 
         WHERE name=p_objname AND owner=p_owner and TYPE=p_objtype
         ORDER BY line;
      
      TYPE ColArray IS TABLE OF GetIdentifiersCursor%rowtype INDEX BY binary_integer;
      TYPE LineArray IS TABLE OF ColArray INDEX BY binary_integer;
   
      LArray LineArray;
      CArray ColArray;
   BEGIN
      
      -- Populate the line and column arrays with identifiers
      FOR Identifier IN GetIdentifiersCursor LOOP

         -- Create a new column array if it doesn't exist
         IF (NOT larray.EXISTS(Identifier.line)) THEN
            larray(Identifier.line) := carray;
         END IF;

         LArray(Identifier.line)(Identifier.col) := Identifier;          
      END LOOP;

      htp.preopen;

      for TextLine in GetTextLineCursor loop
         
         -- Set an anchor for each line to allow us to jump to that line in the 
         -- browser
         htp.prn('<a name=' || TextLine.line || '>'||TextLine.line||'</a>  ');

         IF (NOT larray.EXISTS(TextLine.line))
         THEN
            -- No identifiers were found on this line, so simply print it out
            htp.prn(replace(substr(TextLine.text, 0, length(TextLine.text)-1),  
                    '<', '&#60;'));

            -- Make sure the last character is a newline
            htp.prn(chr(10));
         ELSE
            SrcCol := 1;
            while (SrcCol < length(TextLine.text)) loop

               -- Get the number of non-identifiers
               PrintChars := 0;
               while (LArray(TextLine.line).exists(SrcCol+PrintChars) = FALSE and 
                      SrcCol+PrintChars <= length(TextLine.text)) loop
                 PrintChars := PrintChars+1;
               end loop;

               -- Print out the non-identifier characters
               if (PrintChars > 0) then
                 
                 htp.prn(replace(substr(TextLine.text, SrcCol, PrintChars), '<', 
                         '&#60;'));
                 SrcCol := SrcCol+PrintChars;
               end if;

               -- Print out identifiers
               IF (LArray(TextLine.line).exists(SrcCol))
               THEN
                 -- Insert the proper HTML based on the identifier type and signature
                 IF (LArray(TextLine.line)(SrcCol).signature = p_sig)
                 then
                   htp.prn('<A style="background-color:yellow;text-decoration:none;"
                           HREF=PLScope.DataHit?p_sig='
                           ||LArray(TextLine.line)(SrcCol).signature || 
                           ' target=datawindow>');
                 else
                   htp.prn('<A STYLE="color:'|| 
                           GetTextColor(LArray(TextLine.line)(SrcCol).type) || 
                           ';text-decoration:none;"
                           HREF=PLScope.DataHit?p_sig=' || 
                           LArray(TextLine.line)(SrcCol).signature || 
                           ' target=datawindow>');
                  END IF;

                  -- Print the identifier and increment the SrcCol
                  htp.prn(replace(substr(TextLine.text, SrcCol, 
                                  length(LArray(TextLine.line)(SrcCol).name)), 
                                  '<', '&#60;')||'</A>');
                  SrcCol := SrcCol + length(LArray(TextLine.line)(SrcCol).name);

                  if (SrcCol = length(TextLine.text)) then
                    htp.prn(substr(TextLine.text, SrcCol, 1));
                  end if;
               END IF;
            END LOOP;
         END IF;
         
      END LOOP;

      htp.preclose;
   END DisplaySource;

   PROCEDURE HitBox is
   BEGIN
      FrameTitle('Identifier Lookup Window');

      htp.formopen('PLScope.HitList','GET');

      -- Get the identifier name
      htp.p('<b>Name: </b>');
      htp.formtext('p_name');
      htp.br;

      -- Get the identifier type
      htp.formSelectOpen('p_type', '<b>Type: </b>');
      htp.formSelectOption('ANY');

      -- Use all types available in the all_identifiers view
      for Type in AllTypes.first .. AllTypes.last loop
        htp.formSelectOption(AllTypes(Type));
      end loop;
 
      htp.formSelectClose;

      htp.br;

      htp.formSelectOpen('p_usage', '<b>Has Usage: </b>');

      htp.formSelectOption('DECLARATION');
      htp.formSelectOption('DEFINITION');
      htp.formSelectOption('ASSIGNMENT');
      htp.formSelectOption('CALL');
      htp.formSelectOption('REFERENCE');
 
      htp.formSelectClose;

      htp.formsubmit('p_submit','Identifier Lookup');
      htp.formclose;
   END;
   
   -- Display results of an identifier search
   PROCEDURE HitList(p_name VARCHAR2, p_type varchar2, p_usage varchar2,
                     p_submit varchar2 default NULL) IS
     CurrentName varchar2(30) := Null;
     Found       boolean      := FALSE;
     l_Type      all_identifiers.type%type;
     l_Name      all_identifiers.name%type;

     -- GetIdentCursor is the same as GetIdentDeclCursor except it can search 
     -- for identifiers with a given type and usage.
     cursor GetIdentCursor (p_name VARCHAR2, p_type varchar2, p_usage varchar2) 
     IS
       SELECT *
         FROM all_identifiers
         WHERE (name like p_name or name like upper(p_name)) and
               usage like p_usage and 
               type  like p_type
         ORDER by name, type;

   BEGIN
      if (p_type = 'ANY') then
        l_Type := '%';
      else
        l_Type := p_type;
      end if;

      if (p_name = '' or p_name is null) then
        l_Name := '%';
      else
        l_Name := p_name;
      end if;

      HitBox;
      htp.p('<b>Search results for "' || l_Name || '"</b>');
      htp.br;

      FOR Identifier IN GetIdentCursor(l_Name, l_Type, p_usage) LOOP
         -- Display the name of the current identifier that was found
         if (CurrentName is null or CurrentName != Identifier.name) then
           CurrentName := replace(Identifier.name, '<', '&#60;');
           htp.p('<b>'||CurrentName||':</b>');
           htp.br;
         end if;

         DisplayIdentifierList(Identifier);
         Found := TRUE;
      END LOOP;

      if (Found = FALSE) then
        htp.br;
        htp.p('<b><center STYLE="color:red>No identifiers matched the query</center></b>');
      end if;

   END;
   
END PLScope;
/
show errors;   
      
